
/* This code is a demo of a lightweight approach to interrupt service routines */

#include <avr/io.h>
#include <avr/interrupt.h>
#define F_CPU 8000000UL
#include <util/delay.h>
#define BAUDRATE 9600
#include "USART88.h"
#include "pinDefines.h"



volatile uint8_t buttonIsPressed; /* This variable is defined outside of the main loop
				   and is declared to be volatile -- it can be changed from within any function.*/ 
volatile uint8_t adcValue; /* store ADC value within ISR, send out to serial whenever */

				   
ISR(INT0_vect){				/* lightweight interrupt, takes a microsecond? */
  buttonIsPressed = 1;			/* just set variable and return */
  adcValue = ADCH;
}


void initLED(void){
  // Configure LED, blink to verify
  LED_DDR |= _BV(LED);
  LED_PORT |= _BV(LED);
  _delay_ms(200);
  LED_PORT &=~ _BV(LED);  
}


void initADC(void){
  // Set up ADC on ADC0 / 
  ADMUX |= _BV(ADLAR); /* left-shift for easy 8 bit samples, p. 259 */
  DIDR0 |= _BV(ADC0D); /* turn off digital input buffer on ADC0, p. 260 */
  ADCSRA = _BV(ADEN) | _BV(ADATE) | _BV(ADSC); 
  /* ADC enable, auto-trigger enable, start conversions,  p. 258*/
  ADCSRA |= _BV(ADPS0) | _BV(ADPS1) | _BV(ADPS2);
  /* set ADC clock prescaler do sample every 128 cycles, p. 264,5 */
}


int main(void){
  
  initLED();
  initUART();
  sayOK();
  initADC();

  // Configure interrupts
  EICRA |= _BV(ISC01); 		/* configure interrupt on falling edge, p. 71 */
  EIMSK |= _BV(INT0); 		/* enable interrupt mask for INT0, p. 72 */
  BUTTON_PORT |= _BV(BUTTON);	/* enable pullup */  
  sei();                        /* set interrupts enable */
  
  while(1){			/* mainloop */
    
    if (buttonIsPressed){	/* handle button presses in main loop */
      _delay_ms(50);		/* simple debounce */
      if ((BUTTON_PIN & _BV(BUTTON)) == 0){ /* still low after delay */
	      LED_PORT |= _BV(LED);
	      transmitByte(adcValue);
	      _delay_ms(100);
	      LED_PORT &= ~_BV(LED);	
	
	      buttonIsPressed = 0;	/* remember to reset variable once it's been handled */
      }    
    }
    
  }                             /* end mainloop */
  return(0);
}
